package furny.states;

import furny.ga.FurnLayoutIndividual;
import furny.ga.util.FurnLayoutIOUtil;
import ga.core.GA;
import ga.core.algorithm.interactive.ISIGA;
import ga.core.evaluation.IInteractiveFitnessEvaluator;
import ga.core.individual.IIndividual;
import ga.core.individual.IndividualList;
import ga.view.appstate.SceneState;
import ga.view.appstate.menu.MenuListener;
import ga.view.interfaces.IPhenotypeGenerator;
import ga.view.interfaces.IPostEvaluationState;
import ga.view.interfaces.MouseListener;
import ga.view.processor.OffscreenProcessor;
import ga.view.streaming.nodes.PanelNode;
import ga.view.streaming.nodes.PanelNode.InfoStringType;
import ga.view.streaming.showroom.CameraSettings;
import ga.view.streaming.showroom.ShowRoom;
import ga.view.streaming.showroom.ShowRoomFactory;
import ga.view.streaming.showroom.ShowRoomSettings;
import ga.view.streaming.showroom.ShowRoomSettings.ShowRoomType;

import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.awt.Toolkit;
import java.awt.datatransfer.Clipboard;
import java.awt.datatransfer.StringSelection;
import java.awt.event.ActionEvent;
import java.util.ArrayList;
import java.util.List;
import java.util.Locale;
import java.util.concurrent.Callable;

import javax.swing.AbstractAction;
import javax.swing.JButton;
import javax.swing.JLabel;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.SwingUtilities;

import com.jme3.app.Application;
import com.jme3.app.state.AppStateManager;
import com.jme3.collision.CollisionResult;
import com.jme3.collision.CollisionResults;
import com.jme3.input.MouseInput;
import com.jme3.input.controls.MouseButtonTrigger;
import com.jme3.light.AmbientLight;
import com.jme3.light.DirectionalLight;
import com.jme3.math.Ray;
import com.jme3.math.Vector3f;
import com.jme3.scene.Node;
import com.jme3.scene.Spatial;
import com.jme3.system.AppSettings;

/**
 * Post state to show a summary after the evaluation.
 * 
 * @param <T>
 *          The generic type of the individuals.
 * 
 * @since 12.08.2012
 * @author Stephan Dreyer
 */
public class SummaryPostState<T extends IIndividual<T>> extends SceneState
    implements IPostEvaluationState {

  private static final int INDIVIDUAL_COUNT = 6;

  private SummaryMouseListener summaryMouseListener;

  private final List<PanelNode<T>> menuNodes = new ArrayList<PanelNode<T>>();

  private final IPhenotypeGenerator<T, Node> phenotypeGenerator;

  private final IInteractiveFitnessEvaluator<T> evaluator;
  private final ISIGA<T> algorithm;

  private MenuListener menuListener;

  private SummaryUIState<T> uiState;

  /**
   * Instantiates a new summary post state.
   * 
   * @param evaluator
   *          the evaluator
   * @param algorithm
   *          the algorithm
   * @param phenotypeGenerator
   *          the phenotype generator
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  public SummaryPostState(final IInteractiveFitnessEvaluator<T> evaluator,
      final ISIGA<T> algorithm,
      final IPhenotypeGenerator<T, Node> phenotypeGenerator) {
    this.evaluator = evaluator;
    this.algorithm = algorithm;
    this.phenotypeGenerator = phenotypeGenerator;
  }

  @Override
  public void initialize(final AppStateManager stateManager,
      final Application app) {
    super.initialize(stateManager, app);

    cam.setLocation(new Vector3f(0f, 4.5f, 0f));
    cam.lookAt(Vector3f.ZERO, Vector3f.UNIT_Z.negate());
    cam.setFrustumFar(50);

    inputManager.addMapping("choosePostInd", new MouseButtonTrigger(
        MouseInput.BUTTON_LEFT));

    summaryMouseListener = new SummaryMouseListener();
    inputManager.addListener(summaryMouseListener, "choosePostInd");

    ShowRoomSettings srSettings;

    final Object o = algorithm.getContext().get(GA.KEY_VALIDATION_SPACE);
    if (o != null && o instanceof ShowRoom) {
      srSettings = ((ShowRoom) o).getShowRoomSettings();
    } else {
      srSettings = new ShowRoomSettings();

      srSettings.put(ShowRoomSettings.TYPE, ShowRoomType.BOX);
      srSettings.put(ShowRoomSettings.BOX_WIDTH, 6f);
      srSettings.put(ShowRoomSettings.BOX_LENGTH, 4f);
      srSettings.put(ShowRoomSettings.BOX_HEIGHT, 2.6f);
    }

    final AppSettings settings = app.getContext().getSettings();

    final ShowRoomFactory showRoomFactory = new ShowRoomFactory(assetManager,
        settings, srSettings);

    final IndividualList<T> popIndividuals = new IndividualList<T>(algorithm
        .getPopulation().getIndividuals());
    final IndividualList<T> bestIndividuals = new IndividualList<T>();

    // sort descending to add the better individuals first
    popIndividuals.sort(true);

    for (final T ind : popIndividuals) {
      if (!bestIndividuals.contains(ind)) {
        bestIndividuals.add(ind);

        if (bestIndividuals.size() >= INDIVIDUAL_COUNT) {
          break;
        }
      }
    }

    float x = -2f;
    float z = -1f;

    for (final T ind : bestIndividuals) {
      final ShowRoom showRoom = showRoomFactory.createShowRoom();
      final CameraSettings camSettings = CameraSettings
          .getOrthographicSettings(showRoom, settings);
      final Spatial phenotype = phenotypeGenerator.createPhenotype(ind);
      showRoom.setPhenotype(phenotype);
      final PanelNode<T> node = new PanelNode<T>(assetManager, settings,
          showRoom, camSettings, ind);
      node.setInfoStringType(InfoStringType.COSTS);
      node.setLocalTranslation(new Vector3f(x, 0f, z));
      node.scale(0.65f);
      node.setFadeSpeed(5f);

      final OffscreenProcessor processor = node.getProcessor();

      viewPort.addProcessor(processor);

      menuNodes.add(node);
      rootNode.attachChild(node);

      x += 2f;
      if (x > 2f) {
        x = -2f;
        z = 1f;
      }
    }

    rootNode.addLight(new AmbientLight());
    final DirectionalLight dl = new DirectionalLight();
    dl.setDirection(Vector3f.UNIT_Y.negate());
    rootNode.addLight(dl);

    uiState = new SummaryUIState<T>(evaluator, menuListener);
    stateManager.attach(uiState);
  }

  @Override
  public void setMenuListener(final MenuListener menuListener) {
    this.menuListener = menuListener;
  }

  /**
   * Finds spatials in the scene that has been clicked.
   * 
   * @param node
   *          The parent node to check for clicks.
   * @return The results of the click.
   * 
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  private CollisionResults findPick(final Node node) {
    final Vector3f origin = cam.getWorldCoordinates(
        inputManager.getCursorPosition(), 0.0f);
    final Vector3f direction = cam.getWorldCoordinates(
        inputManager.getCursorPosition(), 0.3f);
    direction.subtractLocal(origin).normalizeLocal();

    final Ray ray = new Ray(origin, direction);
    final CollisionResults results = new CollisionResults();
    node.collideWith(ray, results);
    return results;
  }

  @Override
  public void setEnabled(final boolean enabled) {
    super.setEnabled(enabled);

    // if (viewPort != null) {
    // if (enabled) {
    // if (!renderManager.getMainViews().contains(viewPort)) {
    // viewPort = renderManager.createMainView("Scene", cam);
    // viewPort.setClearFlags(true, true, true);
    // viewPort.attachScene(rootNode);
    // }
    // } else {
    // // DO NOT REMOVE PROCESSORS HERE
    // // while (viewPort.getProcessors().size() > 0) {
    // // SceneProcessor proc = viewPort.getProcessors().get(0);
    // // viewPort.removeProcessor(proc);
    // // }
    //
    // renderManager.removeMainView(viewPort);
    // }
    // }

    if (summaryMouseListener != null) {
      summaryMouseListener.setEnabled(enabled);
    }

    if (uiState != null) {
      uiState.setEnabled(enabled);
    }
  }

  @Override
  public void cleanup() {
    super.cleanup();

    stateManager.detach(uiState);
  }

  /**
   * Mouse listener that detects selection of a summary panel node.
   * 
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  private class SummaryMouseListener extends MouseListener {
    @SuppressWarnings("unchecked")
    @Override
    public void onAction(final String name, final boolean keyPressed,
        final boolean isDoubleClick, final float tpf) {
      if (!keyPressed) {
        final CollisionResults results = findPick(rootNode);

        PanelNode<T> selectedNode = null;

        final CollisionResult r = results.getClosestCollision();
        if (r != null && r.getGeometry() != null
            && r.getGeometry().getParent() != null
            && r.getGeometry().getParent() instanceof PanelNode) {
          selectedNode = (PanelNode<T>) r.getGeometry().getParent();
        }

        if (selectedNode != null) {
          this.setEnabled(false);

          final PanelNode<T> fSelectedNode = selectedNode;

          SwingUtilities.invokeLater(new Runnable() {
            @Override
            public void run() {
              final FurnLayoutIndividual ind = FurnLayoutIndividual.class
                  .cast(fSelectedNode.getIndividual());

              final JPanel panel = new JPanel(new GridBagLayout());
              final GridBagConstraints constraints = new GridBagConstraints();

              constraints.insets = new Insets(5, 5, 5, 5);
              constraints.fill = GridBagConstraints.BOTH;
              constraints.weightx = 1d;
              constraints.gridx = 0;
              constraints.gridy = 0;

              panel.add(new JLabel("Genotype:"), constraints);
              constraints.gridx++;

              panel.add(new JLabel(ind.getGenotypeString()), constraints);

              constraints.gridx++;
              panel.add(new JButton(new AbstractAction("Copy to Clipboard") {
                @Override
                public void actionPerformed(final ActionEvent e) {
                  final StringSelection stringSelection = new StringSelection(
                      ind.getGenotypeString());
                  final Clipboard clipboard = Toolkit.getDefaultToolkit()
                      .getSystemClipboard();
                  clipboard.setContents(stringSelection, null);
                }
              }), constraints);

              constraints.gridx++;
              panel.add(new JButton(new AbstractAction("Save") {
                @Override
                public void actionPerformed(final ActionEvent e) {
                  FurnLayoutIOUtil.saveGenotype(panel, ind);
                }
              }), constraints);

              constraints.gridx = 0;
              constraints.gridy++;

              panel.add(new JLabel("Fitness:"), constraints);

              constraints.gridx++;
              panel.add(
                  new JLabel(String.format(Locale.ENGLISH, "%.2f",
                      ind.getFitness())), constraints);

              constraints.gridx = 0;
              constraints.gridy++;
              panel.add(new JLabel("Costs:"), constraints);

              constraints.gridx++;
              panel.add(new JLabel(ind.getCostString()), constraints);

              JOptionPane.showMessageDialog(null, panel, "Individual details",
                  JOptionPane.PLAIN_MESSAGE);

              app.enqueue(new Callable<Void>() {
                @Override
                public Void call() throws Exception {
                  for (final PanelNode<T> node : menuNodes) {
                    if (node != fSelectedNode) {
                      node.setFadeToAlpha(1f);
                    }
                  }

                  SummaryMouseListener.this.setEnabled(true);

                  return null;
                }
              });

            }
          });

          for (final PanelNode<T> node : menuNodes) {
            if (node != fSelectedNode) {
              node.setFadeSpeed(10f);
              node.setFadeToAlpha(0.2f);
            }
          }
        }
      }
    }
  }

}
